package com.burhanstore.neonbubblequest.Utils.network

import android.util.Log
import com.burhanstore.neonbubblequest.AppConst.Base_URL
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit

object RetrofitClient {

    private val loggingInterceptor = HttpLoggingInterceptor { message ->
        Log.d("API_LOG", message)
    }.apply {
        level = HttpLoggingInterceptor.Level.BODY
    }

    private val userAgentInterceptor = Interceptor { chain ->
        val newRequest = chain.request().newBuilder()
            .header("User-Agent", "RewardPixel/1.0 (Android)")
            .header("Accept", "application/json")
            .build()
        chain.proceed(newRequest)
    }

    // Retry interceptor (optional but recommended)
    private val retryInterceptor = Interceptor { chain ->
        var tryCount = 0
        val request = chain.request()
        var response = chain.proceed(request)

        while (!response.isSuccessful && tryCount < 2) {
            tryCount++
            Log.w("API_RETRY", "Request failed - retrying ($tryCount)...")
            response.close()
            response = chain.proceed(request)
        }

        response
    }

    private val okHttpClient = OkHttpClient.Builder()
        .addInterceptor(userAgentInterceptor)
        .addInterceptor(loggingInterceptor)
        .addInterceptor(retryInterceptor)
        .connectTimeout(90, TimeUnit.SECONDS)
        .readTimeout(90, TimeUnit.SECONDS)
        .writeTimeout(90, TimeUnit.SECONDS)
        .retryOnConnectionFailure(true)
        .build()

    val api: ApiService by lazy {
        Retrofit.Builder()
            .baseUrl(Base_URL)
            .client(okHttpClient)
            .addConverterFactory(GsonConverterFactory.create())
            .build()
            .create(ApiService::class.java)
    }
}
